/* Statistics Generation functions.
 * by Andrew Kempe (TheShadow)
 *     E-mail: <theshadow@shadowfire.org>
 *
 * Services is copyright (c) 1996-1999 Andrew Church.
 *     E-mail: <achurch@dragonfire.net>
 * Services is copyright (c) 1999-2000 Andrew Kempe.
 *     E-mail: <theshadow@shadowfire.org>
 * This program is free but copyrighted software; see the file COPYING for
 * details.
 */

#include "services.h"
#include "pseudo.h"

/*************************************************************************/

#ifdef STATISTICS 

/*************************************************************************/

/* STATISTICS
 *
 *
 *
 * -TheShadow (07 August 1999)
 */

/*************************************************************************/

static ServerStats *serverlist;
static int16 nservers = 0;	/* Number of servers in Server list */

static int16 servercnt = 0;	/* Number of online servers */

/*************************************************************************/

static void do_help(User *u);
static void do_servers(User *u);
static void do_users(User *u);
static void do_map(User *u);

/*************************************************************************/

static Command cmds[] = {
    { "HELP",        do_help,     NULL,  -1,                   -1,-1,-1,-1 },
    { "SERVERS",     do_servers,  NULL,  -1, STAT_HELP_SERVERS,
    		STAT_SERVROOT_HELP_SERVERS,
		STAT_SERVROOT_HELP_SERVERS,
		STAT_SERVROOT_HELP_SERVERS },
    { "USERS",       do_users,    NULL,  STAT_HELP_USERS,      -1,-1,-1,-1 },
    { "MAP",	     do_map,	  NULL,  -1,		       -1,-1,-1,-1 },
    { NULL }
};
 
/*************************************************************************/

static ServerStats *new_serverstats(const char *servername);
static void delete_serverstats(ServerStats *server);

/*************************************************************************/
/****************************** Statistics *******************************/
/*************************************************************************/

/* Return a help message. */

static void do_help(User *u)
{
    char *cmd = strtok(NULL, "");

    if (!cmd) {
	notice_help(s_StatServ, u, STAT_HELP, s_StatServ);
    } else {
	help_cmd(s_StatServ, u, cmds, cmd);
    }
}

/* Main StatServ routine. */

void statserv(const char *source, char *buf)
{
    char *cmd;
    char *s;
    User *u = finduser(source);

    if (!u) {
        log("%s: user record for %s not found", s_StatServ, source);
        notice(s_StatServ, source,
                getstring((NickInfo *)NULL, USER_RECORD_NOT_FOUND));
        return;
    }

    cmd = strtok(buf, " ");
    if (!cmd) {
        return;
    } else if (stricmp(cmd, "\1PING") == 0) {
        if (!(s = strtok(NULL, "")))
            s = "\1";
        notice(s_StatServ, source, "\1PING %s", s);
    } else {
        run_cmd(s_StatServ, u, cmds, cmd);
    }
}

/*************************************************************************/

/* Return information on memory use. Assumes pointers are valid. */

void get_statserv_stats(long *nrec, long *memuse)
{
    ServerStats *server;
    long mem;

    mem = sizeof(ServerStats) * nservers;
    for (server = serverlist; server; server = server->next) {
	mem += strlen(server->name)+1;
	if (server->quit_message)
	    mem += strlen(server->quit_message)+1;
    }

    *nrec = nservers;
    *memuse = mem;
}

/*************************************************************************/
/************************* Server Info Display ***************************/
/*************************************************************************/

/* FIXME
 * 	- Rename all "server" variables to "serverstats".
 */

void do_servers(User *u)
{
    ServerStats *server;
    char *cmd = strtok(NULL, " ");
    char *mask = strtok(NULL, " ");
    struct tm *tm;
    int count = 0;

    if (!cmd)
        cmd = "";

    if (stricmp(cmd, "STATS") == 0) {
	ServerStats *server_lastquit = NULL;
	int onlinecount = 0;
	char lastquit_buf[512];
	
	for (server = serverlist; server; server = server->next) 
	{
	    if (server->t_quit > 0 && (!server_lastquit ||
			    server->t_quit > server_lastquit->t_quit))
		server_lastquit = server;

	    if (server->t_join > server->t_quit)
		onlinecount++;
	}

	notice_lang(s_StatServ, u, STAT_SERVERS_STATS_TOTAL, nservers);
	notice_lang(s_StatServ, u, STAT_SERVERS_STATS_ON_OFFLINE,
			onlinecount, (onlinecount*100)/nservers,
			nservers-onlinecount, 
			((nservers-onlinecount)*100)/nservers);
	
	if (server_lastquit) {
	    tm = localtime(&server_lastquit->t_quit);
	    strftime_lang(lastquit_buf, sizeof(lastquit_buf), u, 
			    STRFTIME_DATE_TIME_FORMAT, tm);
	    notice_lang(s_StatServ, u, STAT_SERVERS_LASTQUIT_WAS,
			    server_lastquit->name, lastquit_buf);
	}


    } else if (stricmp(cmd, "LIST") == 0) {
	int matchcount = 0;

	notice_lang(s_StatServ, u, STAT_SERVERS_LIST_HEADER);
	for (server = serverlist; server; server = server->next) 
	{
	    if (mask && !match_wild_nocase(mask, server->name))
		continue;

	    matchcount++;
	    
	    if (server->t_join < server->t_quit)
		continue;

	    count++;

	    notice(s_StatServ, u->nick, 
			"%-30s %3d (%2d%%)  %3d (%2d%%)",
			server->name, server->usercnt, 
			!usercnt ? 0 : (server->usercnt*100)/usercnt,
			server->opercnt, 
			!opcnt ? 0 : (server->opercnt*100)/opcnt);
	}
	notice_lang(s_StatServ, u, STAT_SERVERS_LIST_RESULTS, 
			count, matchcount);

    } else if (stricmp(cmd, "VIEW") == 0) {
	char *param = strtok(NULL, " ");
	char join_buf[512];
	char quit_buf[512];
	int is_online;
	int limitto = 0;	/* 0 == none; 1 == online; 2 == offline */

	if (param) {
	    if (stricmp(param, "ONLINE") == 0) {
		limitto = 1;
	    } else if (stricmp(param, "OFFLINE") == 0) {
	    	limitto = 2;
	    }
	}

	for (server = serverlist; server; server = server->next) 
	{
	    if (mask && !match_wild_nocase(mask, server->name))
		continue;

	    server->t_join > server->t_quit ? (is_online=1) : (is_online=0);

	    if (limitto && !((is_online && limitto == 1) ||
			    (!is_online && limitto == 2)))
		continue;

	    count++;

	    tm = localtime(&server->t_join);
	    strftime_lang(join_buf, sizeof(join_buf), u, 
				STRFTIME_DATE_TIME_FORMAT, tm);

	    if (server->t_quit != 0) {
		tm = localtime(&server->t_quit);
		strftime_lang(quit_buf, sizeof(quit_buf), u, 
				    STRFTIME_DATE_TIME_FORMAT, tm);
	    } 

	    notice_lang(s_StatServ, u, 
	    		is_online ? STAT_SERVERS_VIEW_HEADER_ONLINE
				  : STAT_SERVERS_VIEW_HEADER_OFFLINE,
			server->name);

	    notice_lang(s_StatServ, u, STAT_SERVERS_VIEW_LASTJOIN, join_buf);
	    if (server->t_quit > 0)
		notice_lang(s_StatServ, u, STAT_SERVERS_VIEW_LASTQUIT, 
				quit_buf);
	    if (server->quit_message)
		notice_lang(s_StatServ, u, STAT_SERVERS_VIEW_QUITMSG,
				server->quit_message);

	    if (is_online) {
		notice_lang(s_StatServ, u, STAT_SERVERS_VIEW_USERS_OPERS,
				server->usercnt, 
				!usercnt ? 0 : (server->usercnt*100)/usercnt,
				server->opercnt, 
				!opcnt ? 0 : (server->opercnt*100)/opcnt);
	    }
	}
	notice_lang(s_StatServ, u, STAT_SERVERS_VIEW_RESULTS, count, nservers);

    } else if (!is_services_root(u)) {
	notice_lang(s_NickServ, u, PERMISSION_DENIED);

    /* Only the Services Root has access from here on! */

    } else if (stricmp(cmd, "DELETE") == 0) {
	ServerStats *serverstats;
	
	if (!mask) {
	    syntax_error(s_StatServ, u, "SERVERS", STAT_SERVERS_DELETE_SYNTAX);

	} else if (!(serverstats = stats_findserver(mask))) {
	    notice_lang(s_StatServ, u, SERV_X_NOT_FOUND, mask);
	    
	} else if (serverstats->t_join > serverstats->t_quit) {
	    notice_lang(s_StatServ, u, STAT_SERVERS_REMOVE_SERV_FIRST, mask);

     	} else {
	    delete_serverstats(serverstats);
	    notice_lang(s_StatServ, u, STAT_SERVERS_DELETE_DONE, mask); 
	}

    } else if (stricmp(cmd, "COPY") == 0) {
	ServerStats *serverstats;
	char *newname = strtok(NULL, " ");

	if (!newname) {
	    syntax_error(s_StatServ, u, "SERVERS", STAT_SERVERS_COPY_SYNTAX);

	} else if (!(serverstats = stats_findserver(mask))) {
	    notice_lang(s_StatServ, u, SERV_X_NOT_FOUND, mask);
	
	} else if (stats_findserver(newname)) {
	    notice_lang(s_StatServ, u, STAT_SERVERS_SERVER_EXISTS, newname);
	    
     	} else {
	    serverstats = new_serverstats(newname);
	    notice_lang(s_StatServ, u, STAT_SERVERS_COPY_DONE, mask, newname);
	}
	
    } else if (stricmp(cmd, "RENAME") == 0) {
	ServerStats *serverstats;
	char *newname = strtok(NULL, " ");

	if (!newname) {
	    syntax_error(s_StatServ, u, "SERVERS", STAT_SERVERS_RENAME_SYNTAX);

	} else if (!(serverstats = stats_findserver(mask))) {
	    notice_lang(s_StatServ, u, SERV_X_NOT_FOUND, mask);
	
	} else if (serverstats->t_join > serverstats->t_quit) {
	    notice_lang(s_StatServ, u, STAT_SERVERS_REMOVE_SERV_FIRST, mask);

	} else if (stats_findserver(newname)) {
	    notice_lang(s_StatServ, u, STAT_SERVERS_SERVER_EXISTS, newname);
	    
     	} else {
	    free(serverstats->name);
	    serverstats->name = sstrdup(newname);
	    notice_lang(s_StatServ, u, STAT_SERVERS_RENAME_DONE, mask, newname);
	}

    } else {
	syntax_error(s_StatServ, u, "SERVERS", STAT_SERVERS_SYNTAX);
    }
}

void do_users(User *u)
{
    char *cmd = strtok(NULL, " ");

    if (!cmd)
        cmd = "";

    if (stricmp(cmd, "STATS") == 0) {
	notice(s_StatServ, u->nick, "         Total users: %d", usercnt);
	notice(s_StatServ, u->nick, "         Total opers: %d", opcnt);
	notice(s_StatServ, u->nick, "Avg users per server: %0.*f", 
			(usercnt > servercnt*2) ? 0 : 2,
			(float)usercnt/servercnt);
	notice(s_StatServ, u->nick, "Avg opers per server: %0.*f", 
			(opcnt > servercnt*2) ? 0 : 2, 
			(float)opcnt/servercnt);
    } else {
	syntax_error(s_StatServ, u, "USERS", STAT_USERS_SYNTAX);
    }
}

int level = 0;
void dump_map(User *u, Server *parent, int length, int longest)
{
    Server *server;
    static char buf[64];

    level++;

    notice(s_StatServ, u->nick, "%s%s", 
    		buf, parent->name);

    server = parent->child;
    if (server) {
	if (length > 0)
	    buf[length-1] = ' ';
	buf[length++] = '|';
	buf[length++] = '-';
	buf[length] = '\0';
    } else {

    }
    while (server) {
	if (server->child) {
	    dump_map(u, server, length, longest);
	} else {
	    notice(s_StatServ, u->nick, "%s%s", 
	    		buf, server->name);
	}
	server = server->sibling;
    }
    length -= 1;
    buf[length] = '\0';
    level--;
}

static void do_map(User *u)
{
    notice(s_StatServ, u->nick, "Command disabled.");
    return;
    dump_map(u, findserver("trinity.shadowfire.org"), 0, 30);
}

/*************************************************************************/
/************************* Statistics Load/Save **************************/
/*************************************************************************/

#define SAFE(x) do {                                    \
    if ((x) < 0) {                                      \
        if (!forceload)                                 \
            fatal("Read error on %s", StatDBName);  \
        nservers = i;                                   \
        break;                                          \
    }                                                   \
} while (0)

void load_ss_dbase()
{
    dbFILE *f;
    int i;
    int16 n;
    int32 tmp32;
    ServerStats *server, *prev;

    if (!(f = open_db(s_StatServ, StatDBName, "r")))
        return;
    switch (i = get_file_version(f)) {
      case 8:
      case 7:
        SAFE(read_int16(&n, f));
        nservers = n;
        if (!nservers) {
            close_db(f);
            return;
        }
	server = NULL;
	prev = NULL;
        for (i = 0; i < nservers; i++) {
	    server = scalloc(sizeof(ServerStats), 1);
	    server->next = prev;
	    prev = server;
            SAFE(read_string(&server->name, f));
	    server->usercnt = 0;
	    server->opercnt = 0;
	    server->t_join = 0;
	    SAFE(read_int32(&tmp32, f));
	    server->t_join = tmp32;
	    SAFE(read_int32(&tmp32, f));
	    server->t_quit = tmp32;
	    SAFE(read_string(&server->quit_message, f));
        }
	serverlist = server;
        break;

      default:
        fatal("Unsupported version (%d) on %s", i, StatDBName);
    } /* switch (ver) */

    close_db(f);
}

#undef SAFE

/*************************************************************************/

#define SAFE(x) do {                                            \
    if ((x) < 0) {                                              \
        restore_db(f);                                          \
        log_perror("Write error on %s", StatDBName);            \
        if (time(NULL) - lastwarn > WarningTimeout) {           \
            wallops(NULL, "Write error on %s: %s", StatDBName,  \
                        strerror(errno));                       \
            lastwarn = time(NULL);                              \
        }                                                       \
        return;                                                 \
    }                                                           \
} while (0)

void save_ss_dbase()
{
    dbFILE *f;
    ServerStats *server;
    static time_t lastwarn = 0;

    if (!(f = open_db(s_StatServ, StatDBName, "w")))
        return;
    SAFE(write_int16(nservers, f));
    for (server = serverlist; server; server = server->next) {
        SAFE(write_string(server->name, f));
	SAFE(write_int32(server->t_join, f));
	SAFE(write_int32(time(NULL), f));
	SAFE(write_string("Services splitting from network.", f));
    }
    close_db(f);
}

#undef SAFE

/*************************************************************************/
/*********************** Internal Stats Functions ************************/
/*************************************************************************/

static ServerStats *new_serverstats(const char *servername)
{
    ServerStats *serverstats;

    nservers++;
    serverstats = scalloc(sizeof(ServerStats), 1);
    serverstats->name = sstrdup(servername);
    serverstats->next = serverlist;
    if (serverstats->next)
	serverstats->next->prev = serverstats;
    serverlist = serverstats;

    return serverstats;
}

/* Remove and free a ServerStats structure. */

static void delete_serverstats(ServerStats *server)
{
    if (debug >= 2)
        log("debug: delete_serverstats() called");

    nservers--;

    if (server->prev)
	server->prev->next = server->next;
    else
	serverlist = server->next;
    if (server->next)
	server->next->prev = server->prev;

    if (debug >= 2)
        log("debug: delete_serverstats(): free ServerStats structure");

    free(server->name);
    free(server->quit_message);
    free(server);

    if (debug >= 2)
        log("debug: delete_serverstats() done");
}

/*************************************************************************/
/*********************** External Stats Functions ************************/
/*************************************************************************/

ServerStats *stats_findserver(const char *servername)
{
    ServerStats *server;

    if (!servername)
	return NULL;

    for (server = serverlist; server; server = server->next) {
	if (stricmp(servername, server->name) == 0) {
	    return server;
	}
    }

    return NULL;
}

ServerStats *stats_do_server(const char *servername, const char *serverhub)
{
    ServerStats *server, *tmpserver;

    server = stats_findserver(servername);

    servercnt++;

    if (server) {
	/* Server has rejoined us */
	
	server->usercnt = 0;
	server->opercnt = 0;

	server->t_join = time(NULL);

    } else {
	/* Totally new server */

	server = new_serverstats(servername);

	server->usercnt = 0;
	server->opercnt = 0;

	server->t_join = time(NULL);
	server->t_quit = 0;
	server->quit_message = NULL;
    }

    if (*serverhub) {
	server->hub = stats_findserver(serverhub);
	if (!server->hub) {
	    /* FIXME: This should NEVER EVER happen - but it's here while this
	     * function is being developed. Remove it someday.  -TheShadow */
	    wallops(s_OperServ, 
		    "WARNING: Could not find server \2%s\2 which is supposed to"
		    "be the hub for \2%s\2", serverhub, servername);
	    log("server: could not find hub %s for %s", serverhub, servername);

	} else {
	    server->child = NULL;
	    if (!server->hub->child) {
		server->hub->child = server;
	    } else {
		tmpserver = server->hub->child;
		while (tmpserver->sibling)
		    tmpserver = tmpserver->sibling;
		tmpserver->sibling = server;
	    }
	}
    }

    return server;
}

/* Handle a user quitting */
void stats_do_quit(const User *user)
{
    ServerStats *serverstats = user->server->stats;

    serverstats->usercnt--;
    if (user->mode & UMODE_O)
        serverstats->opercnt--;
}

/* Handle a server quitting */
void stats_do_squit(const Server *server, const char *quit_message)
{
    ServerStats *serverstats = server->stats;

    serverstats->t_quit = time(NULL);
    if (serverstats->quit_message)
	free(serverstats->quit_message);
    serverstats->quit_message = *quit_message ? strdup(quit_message) : NULL;
    servercnt--;
}

void stats_traverse_tree(ServerStats *parent)
{
    ServerStats *server;

    server = parent->child;
    while (server) {
	server->t_quit = time(NULL);
	if (server->child)
	    stats_traverse_tree(server);
	server = server->sibling;
    }
}

/*************************************************************************/

#endif /* STATISTICS */

/*************************************************************************/
