/* NickServ SENDPASS module.
 *
 * IRC Services is copyright (c) 1996-2007 Andrew Church.
 *     E-mail: <achurch@achurch.org>
 * Parts written by Andrew Kempe and others.
 * This program is free but copyrighted software; see the file COPYING for
 * details.
 */

#include "services.h"
#include "modules.h"
#include "conffile.h"
#include "commands.h"
#include "language.h"
#include "encrypt.h"
#include "modules/mail/mail.h"

#include "nickserv.h"
#include "ns-local.h"

/*************************************************************************/
/*************************** Local variables *****************************/
/*************************************************************************/

static Module *module;
static Module *module_nickserv;
static Module *module_nickserv_mail_auth;
static Module *module_mail;

static time_t NSSendpassDelay = 0;

/*************************************************************************/

static void do_sendpass(User *u);

static Command commands[] = {
    { "SENDPASS", do_sendpass, NULL,              NICK_HELP_SENDPASS, -1,-1 },
    { NULL },
};

/*************************************************************************/
/**************************** Command handler ****************************/
/*************************************************************************/

/* Handle the SENDPASS command. */

static void do_sendpass(User *u)
{
    char *nick = strtok(NULL, " ");
    NickInfo *ni;
    NickGroupInfo *ngi;
    time_t now = time(NULL);
    char passbuf[PASSMAX];
    int res;

    if (!nick) {
	syntax_error(s_NickServ, u, "SENDPASS", NICK_SENDPASS_SYNTAX);
    } else if (!(ni = get_nickinfo(nick))) {
	notice_lang(s_NickServ, u, NICK_X_NOT_REGISTERED, nick);
    } else if (ni->status & NS_VERBOTEN) {
	notice_lang(s_NickServ, u, NICK_X_FORBIDDEN, nick);
    } else if (!(ngi = get_ngi(ni))) {
	notice_lang(s_NickServ, u, INTERNAL_ERROR);
    } else if (!ngi->email) {
	notice_lang(s_NickServ, u, NICK_SENDPASS_NO_EMAIL, nick);
    } else if (ngi->authcode) {
	notice_lang(s_NickServ, u, NICK_SENDPASS_NOT_AUTHED, nick);
    } else if (ngi->last_sendpass
	       && now - ngi->last_sendpass < NSSendpassDelay) {
	notice_lang(s_NickServ, u, NICK_SENDPASS_TOO_SOON,
		    maketime(ngi,NSSendpassDelay-(now-ngi->last_sendpass),0));
    } else if ((res = decrypt(ngi->pass, passbuf, sizeof(passbuf))) != 1) {
	if (res == 0) {
	    module_log("BUG: trapdoor algorithm in use (should have been"
		       " caught at load time)");
	}
	notice_lang(s_NickServ, u, NICK_SENDPASS_FAILED);
    } else {
	char subject[BUFSIZE], body[BUFSIZE];

	snprintf(subject, sizeof(subject),
		 getstring(ngi,NICK_SENDPASS_MAIL_SUBJECT), ni->nick);
	snprintf(body, sizeof(body), getstring(ngi,NICK_SENDPASS_MAIL_BODY),
		 passbuf, s_NickServ, u->username, u->host);
	res = sendmail(ngi->email, subject, body);
	if (res == 0) {
	    ngi->last_sendpass = now;
	    notice_lang(s_NickServ, u, NICK_SENDPASS_SENT, nick);
	} else if (res == 1) {
	    notice_lang(s_NickServ, u, SENDMAIL_NO_RESOURCES);
	} else {
	    module_log("Valid SENDPASS for %s by %s!%s@%s failed",
		       ni->nick, u->nick, u->username, u->host);
	    notice_lang(s_NickServ, u, NICK_SENDPASS_FAILED);
	}
    }
}

/*************************************************************************/
/***************************** Module stuff ******************************/
/*************************************************************************/

const int32 module_version = MODULE_VERSION_CODE;

ConfigDirective module_config[] = {
    { "NSSendpassDelay",  { { CD_TIME, 0, &NSSendpassDelay } } },
    { NULL }
};

/*************************************************************************/

int init_module(Module *module_)
{
    char testbuf[PASSMAX], testbuf2[PASSMAX];


    module = module_;

    if (encrypt("a", 1, testbuf, PASSMAX) < 0) {
	module_log("Couldn't encrypt string to test for trapdoor encryption");
	return 0;
    }
    if (decrypt(testbuf, testbuf2, sizeof(testbuf2)) == 0) {
	module_log("Trapdoor (one-way) encryption in use; SENDPASS cannot"
		   " be used");
	return 0;
    }

    module_nickserv = find_module("nickserv/main");
    if (!module_nickserv) {
	module_log("Main NickServ module not loaded");
	exit_module(0);
	return 0;
    }
    use_module(module_nickserv);

    module_nickserv_mail_auth = find_module("nickserv/mail-auth");
    if (!module_nickserv_mail_auth) {
	module_log("NickServ AUTH module (mail-auth) required for SENDPASS");
	exit_module(0);
	return 0;
    }
    use_module(module_nickserv_mail_auth);

    module_mail = find_module("mail/main");
    if (!module_mail) {
	module_log("Mail module not loaded");
	exit_module(0);
	return 0;
    }
    use_module(module_mail);

    if (!register_commands(module_nickserv, commands)) {
	module_log("Unable to register commands");
	exit_module(0);
	return 0;
    }

    return 1;
}

/*************************************************************************/

int exit_module(int shutdown_unused)
{
#ifdef CLEAN_COMPILE
    shutdown_unused = shutdown_unused;
#endif

    if (module_mail) {
	unuse_module(module_mail);
	module_mail = NULL;
    }
    if (module_nickserv_mail_auth) {
	unuse_module(module_nickserv_mail_auth);
	module_nickserv_mail_auth = NULL;
    }
    if (module_nickserv) {
	unregister_commands(module_nickserv, commands);
	unuse_module(module_nickserv);
	module_nickserv = NULL;
    }

    return 1;
}

/*************************************************************************/
